/*
 * $Id: faum-iso-to-cd.c,v 1.6 2009-10-26 08:51:03 vrsieh Exp $ 
 *
 * gen_FAUM_CD_image: generate a CD-Image read- and writeable by
 * FAUmachine-CD-Writer
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#if 1
#include <assert.h>
#include <fcntl.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "../lib/conv_cd.h"
#include "../lib/simsetup.h"

const char *progname;
const char *in;
const char *out;

const char *basedir = ".";
struct simsetup simsetup;

void time_stop(void) { }
void time_cont(void) { }

#else
static void
usage(int retval)
{
	fprintf(stderr, "%s [OPTION]...\n", progname);
	fprintf(stderr, "Create a custom Hardware-CD-Image.\n\nOptions:\n"
		"\t-size <size>		Useable size of the CD-image in MB\n"
		"\t                      note that the actual image will be bigger!\n"
		"\t                      (use values of 180/650/700/800/850)\n"
		"\t-name <imagename>	Filename of the CD-image\n"
		"\t-type <type>		Type of the CD-image (W/RW)\n"
		"\t-iso  <iso-filename>	Filename of an ISO which will filled into the image\n"
		);
	exit(retval);
}
#endif

static void __attribute__((__noreturn__))
usage(int retval)
{
	fprintf(stderr, "Usage: %s [OPTIONS] <in> <out>\n", progname);
	fprintf(stderr, "\t-h: Show this help.\n");
	exit(retval);
}

int
main(int argc, char **argv)
{
#if 1
	int c;
	void *conv;
	int out_fd;
	uint8_t buf[0x10000];
	unsigned long long pos;
	long long len;
	int ret;

	/*
	 * Get program name.
	 */
	progname = *argv;

	/*
	 * Get options.
	 */
	while ((c = getopt(argc, argv, "h")) != -1) {
		switch (c) {
		case 'h':
			usage(0);
			/*NOTREACHED*/
		default:
			usage(1);
			/*NOTREACHED*/
		}
	}
	argc -= optind;
	argv += optind;

	/*
	 * Get parameter.
	 */
	if (0 < argc) {
		in = *argv;
		argc--;
		argv++;
	} else {
		usage(1);
		/*NOTREACHED*/
	}
	if (0 < argc) {
		out = *argv;
		argc--;
		argv++;
	} else {
		usage(1);
		/*NOTREACHED*/
	}
	if (argc != 0) {
		usage(1);
		/*NOTREACHED*/
	}
	
	/*
	 * Do work.
	 */
	conv = conv_cd_from_iso_open(in);
	assert(c);

	out_fd = open(out, O_WRONLY | O_CREAT, 0666);
	assert(0 <= out_fd);

	for (pos = 0ULL; ; pos += len) {
		len = conv_cd_from_iso_read(conv, buf, sizeof(buf), pos);
		if (len == 0) {
			break;
		}
		ret = write(out_fd, buf, len);
		assert(ret == len);
	}

	ret = close(out_fd);
	assert(0 <= ret);

	conv_cd_from_iso_close(conv);

	return 0;
#else
	int fd;
	int srcfd;
	int ret;

	int32_t medium_size = 0;
	int64_t lead_in_size = 0;
	uint8_t msf[3];
	int option;
	long size = 0;
	const char * filename = NULL;
	const char * isoname = NULL;

	struct option longopts[] = {
		/* name, has_arg, flag, val */
		{ "size",	1, NULL, 's' },
		{ "name",	1, NULL, 'n' },
		{ "type",	1, NULL, 't' },
		{ "iso",	1, NULL, 'i' },
		{ NULL, 0, NULL, 0 }
	};

	progname = *argv;


	/* do some sanity checks, just in case ;) */
	assert(sizeof(mode1) == 2352);
	
	if (argc == 1) {
		fprintf(stderr, "need parameters!\n");
		usage(EXIT_FAILURE);
	}

	while ((option = getopt_long_only(argc, argv, "", longopts, NULL)) != -1) {
		switch(option) {
		case 's':
			size = strtol(optarg, (char **)NULL, 10);
			if (size < 180 || size > 850) {
				fprintf(stderr, "Is there really a sense using"
						"this size? (%ld)\n", size);
				usage(EXIT_FAILURE);
			}
			switch(size) {
			case 180:
				/* http://www.instantinfo.de/detailcdr.php?ID=809
				 *  ATIP-Code: 97m31s07f
				 *  Hersteller: RITEK Corporation
				 *  Dye-Type: Phthalocyanine
				 *  Write-Strategy: 7 - Short Strategy Type
				 *  Datenkapazitaet: 186.47 MB
				 *  Audiokapazitaet: 21.15
				 *  LBA from ATIP: 95475
				 */

				/* maybe insert scsi-3 mmc table 93 p.73 */
				media_desc.start_of_leadin_m = 0x61;
				media_desc.start_of_leadin_s = 0x1F;
				media_desc.start_of_leadin_f = 0x07;
				media_desc.total_blocks = 95475;
				medium_size = media_desc.total_blocks;
				break;
			case 650:
				/* http://www.instantinfo.de/detailcdr.php?ID=729
				 */
				media_desc.start_of_leadin_m = 0x5F;
				media_desc.start_of_leadin_s = 0x1B;
				media_desc.start_of_leadin_f = 0x38;
				media_desc.total_blocks = 333236;
				medium_size = media_desc.total_blocks;
				break;
			case 700:
				/* http://www.instantinfo.de/detailcdr.php?ID=1337
				 */
				media_desc.start_of_leadin_m = 0x61;
				media_desc.start_of_leadin_s = 0x0F;
				media_desc.start_of_leadin_f = 0x11;
				media_desc.total_blocks = 359848;
				medium_size = media_desc.total_blocks;
				break;
			case 800:
				/* http://www.instantinfo.de/detailcdr.php?ID=2022
				 * has invalid ATIP-Values
				 */
				media_desc.start_of_leadin_m = 0x60;
				media_desc.start_of_leadin_s = 0x2B;
				media_desc.start_of_leadin_f = 0x25;
				media_desc.total_blocks = 359849;
				medium_size = 412160;
				break;
			case 850:
				/* http://www.instantinfo.de/detailcdr.php?ID=1400
				 * has invalid ATIP-Values
				 */
				media_desc.start_of_leadin_m = 0x60;
				media_desc.start_of_leadin_s = 0x2B;
				media_desc.start_of_leadin_f = 0x21;
				media_desc.total_blocks = 359849;
				medium_size = 437760;
				break;
			}
			lba_to_msf(media_desc.total_blocks, msf);
			media_desc.last_possible_start_of_leadout_m = msf[0];
			media_desc.last_possible_start_of_leadout_s = msf[1];
			media_desc.last_possible_start_of_leadout_f = msf[2];

			/* add blocks for lead-in */
			msf_to_lba((uint8_t *)&media_desc.start_of_leadin_m, &lead_in_size);
			medium_size += -1 * lead_in_size; /* -1 because of negative block number */
			break;
		case 'n':
			filename = optarg;
			break;
		case 't':
			if (strncmp (optarg, "rw", 2) == 0) {
				media_desc.media_type = 1;
			} else {
				media_desc.media_type = 0;
			}
			break;
		case 'i':
			isoname = optarg;
			break;
		default:
			usage(EXIT_FAILURE);
		}
	}

	if (argc != optind) {
		fprintf(stderr, "unknown parameter: %s\n", argv[optind]);
		usage(EXIT_FAILURE);
	}

	if ( ! filename ) {
		fprintf(stderr, "No FAUM-CD-Image set\n");
		usage(EXIT_FAILURE);
	}

	if ( medium_size == 0 ) {
		fprintf(stderr, "No Medium size set\n");
		usage(EXIT_FAILURE);
	}

	fd = creat(filename, 0666); 
	if (fd <= 0) {
		fprintf(stderr, "opening filename %s failed: %s\n",
						filename, strerror(errno));
		exit(EXIT_FAILURE);
	}

	memcpy(media_desc.magic, "\211FAUM-CDIMAGE", 13);
	media_desc.version = FAUM_IMG_VERSION;
	media_desc.offset = 2048;
	media_desc.toc_size = 4096;
	media_desc.unrestricted_use = 1; /* GEMA-Bit, don't care */
	media_desc.writing_power = 7; /* 12mw */
	media_desc.reference_speed = 3; /* 8x */
	media_desc.disc_status = 0; /* empty */
	media_desc.last_session_status = 0; /* empty */

	/* size must be blocks * 2352 (=block on physical disc) */
	size = medium_size * 2352;

	/* add offset for information */
	size += media_desc.offset;

	/* add blocks for our own toc */
	size += media_desc.toc_size;

	/* create file */
	/* FIXME maybe use glue_storage directly here as well? */
	while (0 < size) {
		unsigned int count;
		static const uint8_t buffer[1024*1024];

		if (size < sizeof(buffer)) {
			count = size;
		} else {
			count = sizeof(buffer);
		}

		ret = xwrite(fd, buffer, count);
		assert(0 < ret);
		assert(ret <= count);

		size -= ret;
	}

	/* set seek pointer to start of file */
	lseek(fd, 0, SEEK_SET);

	if (isoname != NULL) {
		mode1 cd_block;
		int sizeoftrack = 0;
		uint8_t tmp_min;
		uint8_t tmp_sec;
		uint8_t tmp_frame;
		toc_session toc;

		/* set seek pointer to start of data (after offset and lead-in) */
		lseek(fd, media_desc.offset + media_desc.toc_size + (-1 * lead_in_size) + 150 * 2352, SEEK_SET);

		/* open source-file */
		srcfd = open(isoname, O_RDONLY);
		if (srcfd <= 0) {
			fprintf(stderr, "opening filename %s failed: %s\n",
							isoname, strerror(errno));
			exit(EXIT_FAILURE);
		}

		cd_block.typeoftrack = 0x6; /* data, recorded uninterrupted */
		cd_block.datablocktype = 0x8; /* mode 1 */

		/* make sure, image file is not as big as space for payload */
		while ( 1 ) {
			/* read 2048 byte from source and write 2352 to imagefile */
			ret = read(srcfd, cd_block.data, sizeof(cd_block.data));
			if (ret == 0) {
				break;
			}
			ret = write(fd, &cd_block, sizeof(cd_block));
			sizeoftrack++;
			assert(ret);
		}

		close(srcfd);

		media_desc.disc_status = 2; /* complete */
		media_desc.last_session_status = 3; /* complete */
		/* generate TOC */
		tmp_min = media_desc.start_of_leadin_m;
		tmp_sec = media_desc.start_of_leadin_s;
		tmp_frame = media_desc.start_of_leadin_f;

		toc.x01.ctrl = 0x6; /* data, recorded uninterrupted */
		toc.x01.adr = 1;
                toc.x01.tno = 0;
                toc.x01.point = 1;
                toc.x01.min = tmp_min;
                toc.x01.sec = tmp_sec; 
                toc.x01.frame = tmp_frame;
                toc.x01.zero = 0;
                toc.x01.pmin = 0;
                toc.x01.psec = 0;
                toc.x01.pframe = 0;
                toc.x01.blkcount = sizeoftrack;
		toc.x01.datablocktype = 0x8; /* mode 1 */

		/* FIXME whats with the values x02 -> x63 ? */
		/* should they be filled with min sec frame, too? */
		tmp_frame++;
		if ((tmp_frame % 75) == 0) {
			tmp_sec++;
			tmp_frame = 0;
		}
		if ((tmp_sec % 60) == 0) {
			tmp_min++;
			tmp_sec++;
		}

		toc.xA0.ctrl = 0x6; /* data, recorded uninterrupted */
		toc.xA0.adr = 1;
                toc.xA0.tno = 0;
                toc.xA0.point = 0xA0;
                toc.xA0.min = tmp_min; 
                toc.xA0.sec = tmp_sec; 
                toc.xA0.frame = tmp_frame;
                toc.xA0.zero = 0;
                toc.xA0.pmin = 1; /* first track number */
                toc.xA0.psec = media_desc.media_type;
                toc.xA0.pframe = 0;

		tmp_frame++;
		if ((tmp_frame % 75) == 0) {
			tmp_sec++;
			tmp_frame = 0;
		}
		if ((tmp_sec % 60) == 0) {
			tmp_min++;
			tmp_sec++;
		}

		toc.xA1.ctrl = 4;
		toc.xA1.adr = 1;
                toc.xA1.tno = 0;
                toc.xA1.point = 0xA1;
                toc.xA1.min = tmp_min;
                toc.xA1.sec = tmp_sec;
                toc.xA1.frame = tmp_frame;
                toc.xA1.zero = 0;
                toc.xA1.pmin = 1; /* last track number */
                toc.xA1.psec = 0;
                toc.xA1.pframe = 0;

		tmp_frame++;
		if ((tmp_frame % 75) == 0) {
			tmp_sec++;
			tmp_frame = 0;
		}
		if ((tmp_sec % 60) == 0) {
			tmp_min++;
			tmp_sec = 0;
		}

		toc.xA2.ctrl = 4;
		toc.xA2.adr = 1;
                toc.xA2.tno = 0;
                toc.xA2.point = 0xA2;
                toc.xA2.min = tmp_min; 
                toc.xA2.sec = tmp_sec;
                toc.xA2.frame = tmp_frame;
                toc.xA2.zero = 0;
		/* start position of lead-out */
                toc.xA2.pmin = (sizeoftrack + 150) / ( 60 * 75 );
		toc.xA2.psec = (sizeoftrack + 150 - (toc.xA2.pmin * 60 * 75)) % 75;
                toc.xA2.pframe = (sizeoftrack + 150 - (toc.xA2.pmin * 60 * 75) - toc.xA2.psec * 75);

		tmp_frame++;
		if ((tmp_frame % 75) == 0) {
			tmp_sec++;
			tmp_frame = 0;
		}
		if ((tmp_sec % 60) == 0) {
			tmp_min++;
			tmp_sec++;
		}

		/* set seek pointer to start of lead-in */
		lseek(fd, media_desc.offset, SEEK_SET);
		/* write TOC */
		ret = write(fd, &toc, sizeof(toc));
		/* must be completely written */
		assert(ret = sizeof(toc));

		/* set seek pointer to start of file */
		lseek(fd, 0, SEEK_SET);
	}

	/* write media description to file */
	ret = write(fd, &media_desc, sizeof(cd_image));
	/* must be completely written */
	assert(ret = sizeof(cd_image));
	

	close(fd);
	exit(EXIT_SUCCESS);
#endif
}
